/*
 * string.c - syslogd implementation for windows, reference counted strings
 *
 * Created by Alexander Yaworsky
 *
 * THIS SOFTWARE IS NOT COPYRIGHTED
 *
 * This source code is offered for use in the public domain. You may
 * use, modify or distribute it freely.
 *
 * This code is distributed in the hope that it will be useful but
 * WITHOUT ANY WARRANTY. ALL WARRANTIES, EXPRESS OR IMPLIED ARE HEREBY
 * DISCLAIMED. This includes but is not limited to warranties of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <string.h>
#include <windows.h>

#include <glib.h>
#include <glib/gprintf.h>

#include <syslog.h>
#include <syslogd.h>

struct string* string_new( const gchar* init )
{
    struct string *s = g_malloc( sizeof(struct string) );
    s->refcount = 1;
    s->gstr = g_string_new( init );
    return s;
}

struct string* string_new_len( const gchar* init, gssize len )
{
    struct string *s = g_malloc( sizeof(struct string) );
    s->refcount = 1;
    s->gstr = g_string_new_len( init, len );
    return s;
}

struct string* string_addref( struct string* s )
{
    InterlockedIncrement( &s->refcount );
    return s;
}

void string_release( struct string* s )
{
    if( InterlockedDecrement( &s->refcount ) )
        return;

    g_string_free( s->gstr, TRUE );
    g_free( s );
}

gsize string_concat( gchar** result, struct string* s, ... )
{
    va_list args;
    struct string *str;
    gsize length = s->gstr->len;
    gchar *p;

    va_start( args, s );
    while( (str = va_arg( args, struct string* )) )
        length += str->gstr->len;
    va_end( args );

    *result = g_malloc( length + 1 );

    memcpy( *result, s->gstr->str, s->gstr->len );
    p = *result + s->gstr->len;

    va_start( args, s );
    while( (str = va_arg( args, struct string* )) )
    {
        memcpy( p, str->gstr->str, str->gstr->len );
        p += str->gstr->len;
    }
    *p = 0;
    va_end( args );

    return length;
}

struct string* string_vprintf( gchar* fmt, va_list args )
{
    struct string *s;
    gchar *buf;
    int len;

    len = g_vasprintf( &buf, fmt, args );
    s = string_new_len( buf, len );
    g_free( buf );
    return s;
}

struct string* string_printf( gchar* fmt, ... )
{
    struct string *s;
    va_list args;

    va_start( args, fmt );
    s = string_vprintf( fmt, args );
    va_end( args );
    return s;
}

gboolean string_equal( struct string* s1, struct string* s2 )
{
    if( s1 == s2 )
        return TRUE;
    return g_string_equal( s1->gstr, s2->gstr );
}
