/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This is the main module
 */

%{

#define _GNU_SOURCE

#include "config.h"

#include <unistd.h>
#include <stdio.h>

#include "textrope.h"
#include "cfg_file.h"
#include "updatepass.h"
#include "cfg_parser.h"

#define YYPARSE_PARAM param
#define YYLEX_PARAM param

#define param_service_name (((cfg_parser_t *)param)->service_name)
#define param_vars         (((cfg_parser_t *)param)->vars)
#define param_status       (((cfg_parser_t *)param)->status)
#define param_examine      (((cfg_parser_t *)param)->examine)
#define param_error        (((cfg_parser_t *)param)->error)

void
print_error(void *param, const char *msg);

#define yyerror(msg) print_error(param,msg);

%}

%pure_parser

%union {
  textrope_t  *token;
  cfg_field_t *field;
}

%token TOKEN_WS
%token <token> TOKEN_NL
%token <token> TOKEN_IDENTIFIER
%token <token> TOKEN_MYSQL
%token <token> TOKEN_VAR
%token <token> TOKEN_NUMBER
%token <token> TOKEN_OPERATOR
%token <token> TOKEN_EQ
%token <token> TOKEN_EQ_DFLT
%token <token> TOKEN_COMMA
%token <token> TOKEN_DOT
%token <token> TOKEN_APO
%token <token> TOKEN_QUOT
%token <token> TOKEN_ERROR

%start CFG_FILE

%expect 1

%%
CFG_FILE:
		| CFG_LINE
		| CFG_FILE TOKEN_NL
		| CFG_FILE TOKEN_NL CFG_LINE
		;
CFG_LINE:	WSSN
		| WSS TOKEN_IDENTIFIER WSSN TOKEN_MYSQL WSS
		{
		  $<field>$=0;
		  param_status=(textrope_cmp($<token>2,param_service_name)
		                ? STATUS_CONTINUE
		                : update_pass(param_vars,0,0,0));
		  textrope_destroy(&($<token>2));
		  textrope_destroy(&($<token>4));
		  switch (param_status)
		  {
		  case STATUS_CONTINUE:
		    break;
		  case STATUS_DONE:
		    YYACCEPT;
		  default:
		    YYABORT;
		  }
		}
		| WSS TOKEN_IDENTIFIER WSSN TARGET WSSN FIELD TOKEN_COMMA WSS FIELDS
		{
		  char *target;
		  $<field>$=0;
		  if (target=textrope_dup($<token>4))
		    param_status=(textrope_cmp($<token>2,param_service_name)
		                  ? STATUS_CONTINUE
		                  : update_pass(param_vars,target,$<field>6,$<field>9));
		  textrope_destroy(&($<token>2));
		  textrope_destroy(&($<token>4));
		  cfg_field_destroy(&($<field>6));
		  textrope_destroy(&($<token>7));
		  cfg_field_destroy(&($<field>9));
		  if (target) free (target);
		  else
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: short on memory",@4.last_line+1,@4.last_column+1);
		    param_status=ERR_SHORT_MEM;
		    yyerror(0);
		    YYABORT;
		  }
		  switch (param_status)
		  {
		  case STATUS_CONTINUE:
		    break;
		  case STATUS_DONE:
		    YYACCEPT;
		  default:
		    YYABORT;
		  }
		}
		| WSS TOKEN_IDENTIFIER WSSN TARGET WSSN FIELD TOKEN_COMMA WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>2));
		  textrope_destroy(&($<token>4));
		  cfg_field_destroy(&($<field>6));
		  textrope_destroy(&($<token>7));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected field specification(s)",@9.first_line+1,@9.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| WSS TOKEN_IDENTIFIER WSSN TARGET WSSN FIELD error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>2));
		  textrope_destroy(&($<token>4));
		  cfg_field_destroy(&($<field>6));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected comma after primary key specification",@7.first_line+1,@7.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| WSS TOKEN_IDENTIFIER WSSN TARGET WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>2));
		  textrope_destroy(&($<token>4));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected primary key specification",@6.first_line+1,@6.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| WSS TOKEN_IDENTIFIER WSSN TOKEN_MYSQL WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>2));
		  textrope_destroy(&($<token>6));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected end of line",@6.first_line+1,@6.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| WSS TOKEN_IDENTIFIER WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>2));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected target name",@4.first_line+1,@4.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| WSS error
		{
		  $<token>$=0;
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: malformed line",@2.first_line+1,@2.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		;
WSS:
		| WSSN
		;
WSSN:		TOKEN_WS
		| WSSN TOKEN_WS
		;
TARGET:		TOKEN_IDENTIFIER WSS TOKEN_DOT WSS TOKEN_IDENTIFIER
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		  textrope_join(&($<token>$),&($<token>3));
		  textrope_join(&($<token>$),&($<token>5));
		}
		| TOKEN_QUOT
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_IDENTIFIER WSS TOKEN_DOT WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>1));
		  textrope_destroy(&($<token>3));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected table name",@5.first_line+1,@5.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| TOKEN_IDENTIFIER WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>1));
		  textrope_destroy(&($<token>3));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected dot then table name",@3.first_line+1,@3.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		;
FIELDS:		FIELD
		{
		  $<field>$=$<field>1;
		  $<field>1=0;
		}
		| FIELDS TOKEN_COMMA WSS FIELD
		{
		  $<field>$=$<field>1;
		  $<field>1=0;
		  cfg_field_join(&($<field>$),&($<field>4));
		  textrope_destroy(&($<token>2));
		}
		| FIELDS TOKEN_COMMA WSS error
		{
		  $<token>$=0;
		  cfg_field_destroy(&($<field>1));
		  textrope_destroy(&($<token>2));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected field specification",@4.first_line+1,@4.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| FIELDS error
		{
		  $<token>$=0;
		  cfg_field_destroy(&($<field>1));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected comma or newline after field specifiaction",@2.first_line+1,@2.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		;
FIELD:		COLUMN WSS TOKEN_EQ WSS EXPRESSION
		{
		  $<field>$=cfg_field_new($<token>1,$<token>5,FLD_FOR_INSUPD);
		  textrope_destroy(&($<token>1));
		  textrope_destroy(&($<token>3));
		  textrope_destroy(&($<token>5));
		  if (!$<field>$)
		  {
		    cfg_field_destroy(&($<field>$));
		    sprintf(param_error,"Reading config file at %d:%d got error: short on memory",@5.last_line+1,@5.last_column+1);
		    param_status=ERR_SHORT_MEM;
		    yyerror(0);
		    YYABORT;
		  }
		}
		| COLUMN WSS TOKEN_EQ_DFLT WSS EXPRESSION
		{
		  $<field>$=cfg_field_new($<token>1,$<token>5,FLD_FOR_INSERT);
		  textrope_destroy(&($<token>1));
		  textrope_destroy(&($<token>3));
		  textrope_destroy(&($<token>5));
		  if (!$<field>$)
		  {
		    cfg_field_destroy(&($<field>$));
		    sprintf(param_error,"Reading config file at %d:%d got error: short on memory",@5.last_line+1,@5.last_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		    YYABORT;
		  }
		}
		| COLUMN WSS TOKEN_EQ WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>1));
		  textrope_destroy(&($<token>3));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected expression",@5.first_line+1,@5.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| COLUMN WSS TOKEN_EQ_DFLT WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>1));
		  textrope_destroy(&($<token>3));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected expression",@5.first_line+1,@5.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		| COLUMN WSS error
		{
		  $<token>$=0;
		  textrope_destroy(&($<token>1));
		  if (param_status==STATUS_CONTINUE)
		  {
		    sprintf(param_error,"Reading config file at %d:%d got error: expected equal sign",@3.first_line+1,@3.first_column+1);
		    param_status=ERR_CFG_ERR;
		    yyerror(0);
		  }
		  YYABORT;
		}
		;
COLUMN:		TOKEN_IDENTIFIER
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_QUOT
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		;
EXPRESSION:	EXPRESSION_EL
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| EXPRESSION EXPRESSION_EL
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		  textrope_join(&($<token>$),&($<token>2));
		}
		| EXPRESSION WSSN
		{
		  textrope_t *ret;
		  $<token>$=$<token>1;
		  $<token>1=0;
		  if (!textrope_append(&($<token>$),1," "))
		  {
		    textrope_destroy(&($<token>$));
		    sprintf(param_error,"Reading config file at %d:%d got error: short on memory",@2.last_line+1,@2.last_column+1);
		    param_status=ERR_SHORT_MEM;
		    yyerror(0);
		    YYABORT;
		  }
		}
		;
EXPRESSION_EL:	TOKEN_IDENTIFIER
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_MYSQL
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_VAR
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_NUMBER
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_OPERATOR
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_DOT
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_APO
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		| TOKEN_QUOT
		{
		  $<token>$=$<token>1;
		  $<token>1=0;
		}
		;

%%

void
print_error(void *param, const char *msg)
{
  fprintf(stderr,"%s\n", *param_error ? param_error : msg);
  if (param_examine)
  {
    textrope_destroy(param_examine);
    param_examine=0;
  }
}

int
cfg_parser(const char *service_name, variable_t **vars)
{
  int           ret;
  cfg_parser_t  buff;
  extern FILE  *yyin;
  buff.service_name=service_name;
  buff.vars=vars;
  buff.status=STATUS_CONTINUE;
  buff.examine=0;
  *buff.error=0;
  if (!(yyin=fopen(CFG_FILE,"r")))
  {
    fprintf(stderr,"Cannot open configuration file: \"%s\"\n", CFG_FILE);
    return ERR_CFG_ERR;
  }
  ret=yyparse(&buff);
  fclose(yyin);
  yyin=0;
  if (buff.examine)
  {
    textrope_destroy(buff.examine);
    buff.examine=0;
  }
  switch (buff.status)
  {
  case STATUS_DONE:
    if (!ret) return 0;
  case STATUS_CONTINUE:
    if (!ret) fprintf(stderr,"Cannot find service %s in configuration file\n",
                      service_name);
    return ERR_CFG_ERR;
  }
  return buff.status;
}
