/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This is the main module
 */

#include "config.h"

#include <mysqld_error.h>
#include <stdio.h>
#include <string.h>

#include "textrope.h"
#include "db_mysql.h"

static int
mysql_query_tr(MYSQL *db, textrope_t **cmd, int *perr)
{
  char *str;
  int   err, ret=STATUS_CONTINUE;
  if (!(str=textrope_dup(*cmd))) ret=ERR_SHORT_MEM;
  if (!perr) perr=&err;
  if ((ret==STATUS_CONTINUE) && (*perr=mysql_query(db,str))) ret=ERR_DB;
  if (str)
  {
    bzero(str,strlen(str));
    free(str);
  }
  return ret;
}

int
db_mysql_open(MYSQL *db, const char *root_passwd)
{
  mysql_init(db);
  if (!mysql_real_connect(db,0,"root",root_passwd,0,0,0,0)) {
    fprintf(stderr,"Cannot connect to server\n");
    return ERR_DB;
  }
  return STATUS_CONTINUE;
}

void
db_mysql_close(MYSQL *db)
{
  mysql_close(db);
}

int
db_mysql_db_passwd(MYSQL *db, const char *user, const char *passwd,
                   int verbose)
{
  textrope_t *cmd=0;
  int         ret=STATUS_CONTINUE;
  if (mysql_select_db(db,user)) switch (mysql_errno(db))
  {
  case ER_BAD_DB_ERROR:
    if (!textrope_append(&cmd,0,"create database ")) ret=ERR_SHORT_MEM;
    if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,user))
      ret=ERR_SHORT_MEM;
    if (ret==STATUS_CONTINUE) ret=mysql_query_tr(db,&cmd,0);
    textrope_purge(&cmd);
    if (ret!=STATUS_CONTINUE)
    {
      fprintf(stderr,"Cannot create database %s\n",user);
      return ret;
    }
    else if (verbose) fprintf(stderr,"Database %s sucessfully created\n",user);
  case 0:
    break;
  default:
    fprintf(stderr,"Database connection error #%d : %s\n",mysql_errno(db),
            mysql_error(db));
    return ERR_DB;
  }
  if (!textrope_append(&cmd,0,"grant usage on *.* to ")) ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,user))
      ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,"@localhost identified by \'"))
      ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_esc(&cmd,passwd,'\\',"\\\'\""))
      ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,"\'"))
      ret=ERR_SHORT_MEM;
  if (ret==STATUS_CONTINUE) ret=mysql_query_tr(db,&cmd,0);
  textrope_purge(&cmd);
  if (ret!=STATUS_CONTINUE)
  {
    fprintf(stderr,"Cannot change password\n");
    return ret;
  }
  else if (verbose) fprintf(stderr,"Password changed.\n");
  if (!textrope_append(&cmd,0,"grant all on ")) ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,user))
      ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,".* to "))
      ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,user))
      ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !textrope_append(&cmd,0,"@localhost"))
      ret=ERR_SHORT_MEM;
  if (ret==STATUS_CONTINUE) ret=mysql_query_tr(db,&cmd,0);
  textrope_purge(&cmd);
  if (ret!=STATUS_CONTINUE)
  {
    fprintf(stderr,"Cannot grant privileges\n");
    return ret;
  }
  else if (verbose)
    fprintf(stderr,"Supervisor privileges to database %s granted.\n",user);
  return STATUS_CONTINUE;
}

static int
sql_list_append(textrope_t **cmd, cfg_field_t *fields, const char *sep_pair,
                const char *sep_list, int lstatus, int rstatus)
{
  int sep=0, l, r;
  while (fields)
  {
    l=(lstatus & fields->status);
    r=(rstatus & fields->status);
    if (l || r)
    {
      if (sep && !textrope_append(cmd,0,sep_list)) return 0;
      sep=1;
    }
    if (l && !textrope_append(cmd,0,fields->lvalue)) return 0;
    if (l && r && !textrope_append(cmd,0,sep_pair)) return 0;
    if (r && !textrope_append(cmd,0,fields->rvalue)) return 0;
    fields=fields->next;
  }
  return 1;
}

static int
sql_insert(textrope_t **cmd, const char *target, cfg_field_t *primary,
           cfg_field_t *fields)
{
  if (!textrope_append(cmd,0,"insert into ")) return 0;
  if (!textrope_append(cmd,0,target)) return 0;
  if (!textrope_append(cmd,0," (")) return 0;
  if (!sql_list_append(cmd,primary,0,", ",FLD_FOR_INSUPD,0)) return 0;
  if (!textrope_append(cmd,0,", ")) return 0;
  if (!sql_list_append(cmd,fields,0,", ",FLD_FOR_INSERT,0)) return 0;
  if (!textrope_append(cmd,0,") values (")) return 0;
  if (!sql_list_append(cmd,primary,0,", ",0,FLD_FOR_INSUPD)) return 0;
  if (!textrope_append(cmd,0,", ")) return 0;
  if (!sql_list_append(cmd,fields,0,", ",0,FLD_FOR_INSERT)) return 0;
  if (!textrope_append(cmd,0,")")) return 0;
  return 1;
}

static int
sql_update(textrope_t **cmd, const char *target, cfg_field_t *primary,
           cfg_field_t *fields)
{
  if (!textrope_append(cmd,0,"update ")) return 0;
  if (!textrope_append(cmd,0,target)) return 0;
  if (!textrope_append(cmd,0," set ")) return 0;
  if (!sql_list_append(cmd,fields,"=",", ",FLD_FOR_UPDATE,FLD_FOR_UPDATE))
    return 0;
  if (!textrope_append(cmd,0," where ")) return 0;
  if (!sql_list_append(cmd,primary,"="," and ",FLD_FOR_INSUPD,FLD_FOR_INSUPD))
    return 0;
  return 1;
}

static int
sql_select(textrope_t **cmd, const char *target, cfg_field_t *primary)
{
  if (!textrope_append(cmd,0,"select count(*) from ")) return 0;
  if (!textrope_append(cmd,0,target)) return 0;
  if (!textrope_append(cmd,0," where ")) return 0;
  if (!sql_list_append(cmd,primary,"="," and ",FLD_FOR_INSUPD,FLD_FOR_INSUPD))
    return 0;
  return 1;
}

int
db_mysql_table_passwd(MYSQL *db, const char *target, cfg_field_t *primary,
                      cfg_field_t *fields, int verbose)
{
  textrope_t *cmd=0;
  int         ret=STATUS_CONTINUE, num;
  MYSQL_ROW   row=0;
  MYSQL_RES  *res=0; 
  if (!sql_select(&cmd,target,primary)) ret=ERR_SHORT_MEM;
  if (ret==STATUS_CONTINUE) ret=mysql_query_tr(db,&cmd,0);
  textrope_purge(&cmd);
  if ((ret==STATUS_CONTINUE) && !(res=mysql_store_result(db))) ret=ERR_SHORT_MEM;
  if ((ret==STATUS_CONTINUE) && !(row=mysql_fetch_row(res))) ret=ERR_DB;
  if ((ret==STATUS_CONTINUE) && (sscanf(row[0],"%d",&num)!=1)) ret=ERR_DB;
  if (res) mysql_free_result(res);
  if (ret==STATUS_CONTINUE) switch (num)
  {
  case 0:
    if (!sql_insert(&cmd,target,primary,fields)) ret=ERR_SHORT_MEM;
    if (ret==STATUS_CONTINUE) ret=mysql_query_tr(db,&cmd,0);
    textrope_purge(&cmd);
    if (ret!=STATUS_CONTINUE)
      fprintf(stderr,"Cannot insert user record to table %s\n",target);
    else if (verbose) fprintf(stderr,"User record added to table %s.\n",target);
    break;
  case 1:
    if (!sql_update(&cmd,target,primary,fields)) ret=ERR_SHORT_MEM;
    if (ret==STATUS_CONTINUE) ret=mysql_query_tr(db,&cmd,0);
    textrope_purge(&cmd);
    if (ret!=STATUS_CONTINUE)
      fprintf(stderr,"Cannot update user record in table %s\n",target);
    else if (verbose)
      fprintf(stderr,"User recod updated in table %s with new password.\n",
              target);
    break;
  default:
    fprintf(stderr,"Primary key is not unique in table %s\n",target);
    return ret;
  }
  else fprintf(stderr,"Cannot browse table %s\n",target);
  return ret;
}
