/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This is the main module
 */

#define _GNU_SOURCE

#include "config.h"

#include <stdio.h>
#include <getopt.h>
#include <pwd.h>
#include <string.h>

/*
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
*/

#include "cfg_parser.h"
#include "vars.h"

#ifndef MYSQLPASSWD_BUGREPORT
#  define MYSQLPASSWD_BUGREPORT "Bartosz Lis <bartoszl@icsp.lodz.pl>"
#endif /*MYSQLPASSWD_BUGREPORT */
#ifndef MYSQLPASSWD_STRING
#  define MYSQLPASSWD_STRING PACKAGE " " VERSION
#endif /*MYSQLPASSWD_STRING*/

char *progname;

void
error_msg (FILE *f, const char *msg, int usage, int code)
{
   if (msg) fprintf(f, "%s: %s\n",progname,msg);
   else fprintf(f, MYSQLPASSWD_STRING "\n"
                   "Changing mysql passwords\n"
                   "Copyright (C) 2002 " MYSQLPASSWD_BUGREPORT "\n"
                   "This program is distributed under the GNU General Public License\n",
                   "For more information see the file COPYING in its sources\n");
   if (usage)
     {
        fprintf(f, "\nUsage: %s [-h | -v | { [-s <service>] [-u <user>]}\n"
                   "where: -h, --help                 -- help screen\n"
                   "       -v, --version              -- version\n"
                   "       -u, --user <user>          -- change <user> password\n"
                   "                                     this option is valid for root only\n"
                   "       -s, --service <service>    -- which service's password to alter\n",
                progname);
     }
   fflush(0);
   switch (code)
   {
   case STATUS_CONTINUE:
   case STATUS_DONE:
     code=0;
   default:
     exit(-code);
   }
}

struct option options[] =
{
  { "help",       no_argument,       0, 'h' },
  { "version",    no_argument,       0, 'v' },
  { "user",       required_argument, 0, 'u' },
  { "service",    required_argument, 0, 's' },
  { 0, 0, 0, 0 }
};

struct optval {
  const char *user;
  const char *service;
  uid_t       uid;
};

void
opt_init(struct optval *opt)
{
  opt->user=0;
  opt->service=0;
}

void
opt_parse(struct optval *opt, int argc, char **argv)
{
  char     c;
  unsigned x;
  progname=argv[0];
  opt_init(opt);
  while ((c=getopt_long(argc,argv,"hvu:s:",options,0))!=-1) switch (c)
  {
  case 'u':
    opt->user=strdup(optarg);
    break;
  case 's':
    opt->service=strdup(optarg);
    break;
  case 'h':
    error_msg(stdout,0,1,STATUS_CONTINUE);
  case 'v':
    error_msg(stdout,0,0,STATUS_CONTINUE);
  case ':':
    error_msg(stderr,"missing parameter for an option",1,ERR_PARAM);
  default:
    error_msg(stderr,"unknown option",1,ERR_PARAM);
  }
  if (optind<argc) error_msg(stderr,"unknown argument",1,ERR_PARAM);
  if (!opt->service) if (opt->service=rindex(progname,'/')) ++(opt->service);
  else opt->service=progname;
}

void
opt_check(struct optval *opt)
{
  struct passwd *pwd;
  char          *forbidden[] = FORBIDDEN_LOGINS;
  char         **ptr;
  size_t         len;
  if(!opt->service || !*opt->service)
    error_msg(stderr,"wrong service specified",0,ERR_PARAM);
  if (opt->uid=getuid())
  {
    if (opt->uid<MIN_UID)
      error_msg(stderr,"user is not allowed to run this program",0,ERR_PARAM);
    if (!(pwd=getpwuid(opt->uid)))
      error_msg(stderr,"cannot find current user",0,ERR_PARAM);
    if (opt->user)
      error_msg(stderr,"only root can change other user passwords",0,ERR_PARAM);
    opt->user=strdup(pwd->pw_name);
  }
  else if (!opt->user)
    error_msg(stderr,"user must be explicitly specified ",0,ERR_PARAM);
  if(!getpwnam(opt->user))
    error_msg(stderr,"unknown user specified",0,ERR_PARAM);
  for (ptr=forbidden; *ptr; ++ptr)
  {
    len=strlen(*ptr);
    if (len) if (!(((*ptr)[len-1])=='*' ? strncmp(*ptr,opt->user,len-1)
                                        : strcmp(*ptr,opt->user)))
      error_msg(stderr,"this user's password is not allowed to be changed "
                       "with this program",0,ERR_PARAM);
  }
}

int main(int argc, char **argv)
{
  struct optval  opt;
  int            ret;
  variable_t    *vars=0;
  char           buff[20];
  opt_parse(&opt,argc,argv);
  opt_check(&opt);
  sprintf(buff,"%d",opt.uid);
  set_var(&vars,VAR_LOGIN,opt.user);
  set_var(&vars,VAR_UID,buff);
  set_var(&vars,VAR_SERVICE,opt.service);
  ret=cfg_parser(opt.service,&vars);
  purge_vars(&vars);
  return ret;
}
