/***************************************************************************
                          ann_array.h  -  description
                             -------------------
    begin                : pon kwi 14 2003
    copyright            : (C) 2003 by Bartosz Lis
    email                : bartoszl@ics.p.lodz.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __ANN_ARRAY_H
#define __ANN_ARRAY_H

#include <ctype.h>
#include <math.h>

#include <ann_nav.h>
#include <ann_manip.h>
#include <ann_loc.h>

namespace ANN
{

  template<class Elem, class Item=Elem, class Manip=Manip_val<Elem> >
  class Array; /* weak container */

  template<class Elem, class Item=Elem, class Manip=Manip_val<Elem> >
  class ARRAY; /* strong container */

  //template<class Elem, class Item, class Manip>//
 // class Array                                 //
//                                             //

  template<class Elem, class Item, class Manip>
  class Array
  {
  public:
    typedef Elem  Elem_t;
    typedef Item  Item_t;
    typedef Manip Manip_t;
    
  protected:
    Size           size;
    Elem          *data;
    static Elem    nav;

    virtual bool do_copy_data(size_t size_, const Item_t *data_);
    virtual void do_set_data(Elem_t *data_);
    
  public:
    virtual bool resize(const Size &limits, bool keep=true);
    bool         copy_data(size_t size_, const Item_t *data_)
    { return do_copy_data(size_,data_); }
    void         copy_data(const Item_t *data_)
    { do_copy_data(size.total_size(),data_); }
    void         set_data(Elem_t *data_)
    { do_set_data(data_); }

    Array() : data(0) {}
    Array(const Array &that) : size(that.size), data(that.data) {}
    Array(const Size &size_, Elem_t *data_=0) : size(size_), data(data_) {}

    virtual ~Array();

    const Size   &get_sizes() const { return size; }
    size_t        get_size() const { return size.total_size(); }
    Elem_t       &at(size_t idx)
    { return (idx<size.total_size()) && data ? data[idx] : NAV_set(nav); }
    Item_t        at(size_t idx) const
    { return (idx<size.total_size()) && data ? data[idx] : NAV_get(Elem); }
    Elem_t       &at(const Loc &loc) { return at(size.index(loc)); }
    Item_t        at(const Loc &loc) const { return at(size.index(loc)); }
    Elem_t       *get_data() { return data; }
    const Item_t *get_data() const { return data; }

    Elem_t &operator [] (size_t idx) { return at(idx); }
    Item_t  operator [] (size_t idx) const { return at(idx); }
    Elem_t &operator [] (const Loc &loc) { return at(loc); }
    Item_t  operator [] (const Loc &loc) const { return at(loc); }
    Array  &operator = (const Array &arr)
    { resize(arr.size,false); set_data(arr.data); return *this; }
  };

  //template<class Elem, class Item, class Manip>//
 // class ARRAY                                 //
//                                             //

  template<class Elem, class Item, class Manip>
  class ARRAY : public Array<Elem,Item,Manip>
  {
  public:
    typedef Elem  Elem_t;
    typedef Item  Item_t;
    typedef Manip Manip_t;

  protected:

    virtual void do_set_data(Elem_t *data_);

  public:
    virtual bool resize(const Size &limits, bool keep=true);

    ARRAY() {}
    ARRAY(const Array<Elem,Item> &that)
    {
      resize(that.get_sizes(),false);
      if (that.get_data()) copy_data(that.get_data());
    }
    ARRAY(const Size &size_, const Item_t *data_)
    { resize(size_,false); if (data_) copy_data(data_); }
    ARRAY(const Size &size_)
    { resize(size_,false); }

    virtual ~ARRAY();

    ARRAY  &operator = (const Array<Elem,Item> &arr)
    {
      resize(arr.get_sizes(),false);
      if (arr.get_data()) copy_data(arr.get_data());
      return *this;
    }
  };

  /*template<class E, class I>
  static inline void clone_elem(E &e, I i) { e=i; }
  template<class E>
  static inline void clone_elem(E *&e, const E *i) { e=new E(*i); }
  template<class E, class I>
  static inline void clear_array(E *arr, I nan, size_t size) {}
  template<class E>
  static inline void clear_array(E **arr, const E *nan, size_t size)
  { for (size_r j=0; j<size; ++j) if (arr[j]!=i) }*/
  
} // namespace ANN

  //template<class Elem, class Item, class Manip>//
 // class Array                                 //
//  inlines                                    //

template<class Elem, class Item, class Manip>
Elem ANN::Array<Elem,Item,Manip>::nav;

template<class Elem, class Item, class Manip>
ANN::Array<Elem,Item,Manip>::~Array()
{
}

template<class Elem, class Item, class Manip>
bool
ANN::Array<Elem,Item,Manip>::do_copy_data(size_t size_, const Item *data_)
{
  size_t i;
  if (!data_) size_=0;
  else if (size_>size.total_size()) size_=size.total_size();
  Manip::clear_array(data,size.total_size());
  for (i=0; i<size_; ++i) Manip::clone(data[i],data_[i]);
  for (i=size_; i<size.total_size(); ++i) NAV_set(data[i]);
  return true;
}

template<class Elem, class Item, class Manip>
void
ANN::Array<Elem,Item,Manip>::do_set_data(Elem *data_)
{
  data=data_;
}

template<class Elem, class Item, class Manip>
bool
ANN::Array<Elem,Item,Manip>::resize(const Size &limits, bool keep)
{
  bool ret=(size.total_size()==limits.total_size());
  if (!ret && !keep) data=0;
  size=limits;
  return ret;
}

  //template<class Elem, class Item, class Manip>//
 // class ARRAY                                 //
//  inlines                                    //

template<class Elem, class Item, class Manip>
ANN::ARRAY<Elem,Item,Manip>::~ARRAY()
{
  if (data) delete [] data;
}

template<class Elem, class Item, class Manip>
void
ANN::ARRAY<Elem,Item,Manip>::do_set_data(Elem *data_)
{
  copy_data(data_);
}

template<class Elem, class Item, class Manip>
bool
ANN::ARRAY<Elem,Item,Manip>::resize(const Size &limits, bool keep)
{
  size_t i, n=size.total_size(), n_=limits.total_size();
  bool   ret=(n==n_);
  if (!ret)
  {
    Elem *tmp=((n_) ? new Elem[n_] : 0);
    if (keep)
    {
      if (n>n_) n=n_;
      for (i=0; i<n; ++i) tmp[i]=data[i];
    }
    else n=0;
    Manip::clear_array(data+n,size.total_size()-n);
    for (i=n; i<n_; ++i) NAV_set(tmp[i]);
    if (data) delete [] data;
    data=tmp;
  }
  size=limits;
  return ret;
}

#endif /* __ANN_ARRAY_H */
