/***************************************************************************
                          ann_loc.h  -  description
                             -------------------
    begin                : pon kwi 14 2003
    copyright            : (C) 2003 by Bartosz Lis
    email                : bartoszl@ics.p.lodz.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __ANN_LOC_H
#define __ANN_LOC_H

#include <ctype.h>

namespace ANN
{

  class Loc;  /* position pointer to multidimensional arrays */
  class Size; /* Size of multidimensional arrays */

  const size_t size_max=size_t(-1);

  //           //
 // class Loc //
//           //

  class Loc
  {
  protected:
    static size_t  nav;
    size_t         dimension;
    union
    {
      size_t       coordinate;
      size_t      *coordinates;
    }              pos;
    size_t         cap;

    static size_t get_dim(const size_t *coordinates_);
    void          resize(size_t dimension_, const size_t *coordinates_,
                         size_t value);
    
  public:
    void          resize(size_t dimension_, const size_t *coordinates_)
    { resize(dimension_,coordinates_,0); }
    void          resize(size_t dimension_, size_t value=0)
    { resize(dimension_,0,value); }
    void          resize(const size_t *coordinates_=0)
    { resize(get_dim(coordinates_),coordinates_,0); }
    void          resize(const Loc &that)
    { resize(that.dimension,that.coord(),0); }
    inline void   resize(const Size &that);

    Loc() : dimension(0), cap(0) {}
    Loc(const Loc &that) : dimension(0), cap(0) { resize(that); }
    Loc(const Size &that) : dimension(0), cap(0) { resize(that); }
    Loc(size_t dimension_, const size_t *coordinates_)
    : dimension(0), cap(0) { resize(dimension_,coordinates_,0); }
    Loc(size_t dimension_, size_t value)
    : dimension(0), cap(0) { resize(dimension_,0,value); }
    Loc(size_t coordinate_)
    : dimension(1), cap(0) { pos.coordinate=coordinate_; }
    Loc(const size_t *coordinates_)
    : dimension(0), cap(0) { resize(coordinates_); }

    ~Loc();

    size_t        dim() const { return dimension; }
    const size_t *coord() const
    { return cap ? pos.coordinates : (dimension==1 ? &pos.coordinate : 0); }
    size_t       *coord()
    { return cap ? pos.coordinates : (dimension==1 ? &pos.coordinate : 0); }
    size_t        product() const; /* multiplication of all coordinates */
    Loc          &expand(const Loc &that);
    /* expands each coords, so each of them is max of its previous value and
     * value of that */
    Loc          &shrink(const Loc &that);
    Loc          &shrink_back(const Loc &that);
    Loc          &append(size_t dim_, const size_t *coord_);
    Loc          &append(size_t coord_) { return append(1,&coord_); }
    Loc          &append(const Loc &that)
    { return append(that.dim(),that.coord()); }
    void          clear(); /* zeroes all coordinates */
    size_t       &at(size_t which_dim)
    { return which_dim<dimension ? coord()[which_dim] : nav=size_max; }
    size_t        at(size_t which_dim) const
    { return which_dim<dimension ? coord()[which_dim] : size_max; }
    short         cmp(const Loc &that) const;
    
    size_t &operator [] (size_t which_dim) { return at(which_dim); }
    size_t  operator [] (size_t which_dim) const { return at(which_dim); }
    Loc    &operator = (const Loc &that) { resize(that); return *this; }
    bool    operator == (const Loc &that) const  { return !cmp(that); }
    bool    operator != (const Loc &that) const  { return cmp(that); }
    bool    operator < (const Loc &that) const  { return cmp(that)<0; }
    bool    operator <= (const Loc &that) const  { return cmp(that)<=0; }
    bool    operator > (const Loc &that) const  { return cmp(that)>0; }
    bool    operator >= (const Loc &that) const  { return cmp(that)>=0; }
    Loc    &operator << (size_t coord_) { return append(coord_); }
    Loc    &operator << (const Loc &that) { return append(that); }
  };

  //            //
 // class Size //
//            //

  class Size : private Loc
  {
  protected:
    size_t total;
    
    size_t *coord() { return Loc::coord(); }    
    void    resize(size_t dimension_, const size_t *coordinates_,
                   size_t value);
    
  public:
    void    resize(size_t dimension_, const size_t *coordinates_)
    { resize(dimension_,coordinates_,0); }
    void    resize(size_t dimension_, size_t value)
    { resize(dimension_,0,value); }
    void    resize(size_t total_) { Loc::resize(1,&total_,0); total=total_; }
    void    resize(const size_t *coordinates_=0) 
    { resize(get_dim(coordinates_),coordinates_,0); }
    void    resize(const Loc &that) { resize(that.dim(),that.coord(),0); }
    void    resize(const Size &that)
    { Loc::resize(that.dimension,that.coord(),0); total=that.total; }
          
    Size() : total(0) {}
    Size(const Loc &that) : total(0) { resize(that); }
    Size(const Size &that) : Loc(), total(0) { resize(that); }
    Size(size_t dimension_, const size_t *coordinates_)
    : total(0) { resize(dimension_,coordinates_,0); }
    Size(size_t dimension_, size_t value)
    : total(0) { resize(dimension_,0,value); }
    Size(size_t total_) : Loc(total_), total(total_) {}
    Size(const size_t *coordinates_) : total(0) { resize(coordinates_); }

    ~Size() {}

    size_t        dim() const { return dimension; }
    const size_t *coord() const { return Loc::coord(); }
    size_t        total_size() const { return total; }
    Size         &expand(const Loc &that);
    Size         &expand(const Size &that);
    Size         &shrink(const Size &that);
    Size         &shrink_back(const Size &that);
    Size         &append(size_t dim_, const size_t *coord_);
    Size         &append(size_t coord_)
    { set(dimension,coord_); return *this; }
    Size         &append(const Size &that);
    size_t        index(const Loc &loc) const;
    bool          next(Loc &loc) const;
    bool          locate(size_t idx, Loc &loc) const;
    /* flat index of loc using *this as limits */
    void          reset();
    size_t        at(size_t which_dim) const { return Loc::at(which_dim); }
    void          set(size_t which_dim, size_t size_);
   
    size_t  operator [] (size_t which_dim) const { return at(which_dim); }
    size_t  operator [] (const Loc &loc) const { return index(loc); }
    Size   &operator = (const Loc &that) { resize(that); return *this; }
    Size   &operator = (const Size &that) { resize(that); return *this; }
    bool    operator == (const Size &that) const  { return !Loc::cmp(that); }
    bool    operator != (const Size &that) const  { return Loc::cmp(that); }
    Size   &operator << (size_t coord_) { return append(coord_); }
    Size   &operator << (const Size &that) { return append(that); }
  };

} // namespace ANN

inline void 
ANN::Loc::resize(const Size &that)
{ 
  resize(that.dim(),that.coord(),0);
}

#endif /* __ANN_LOC_H */
