/***************************************************************************
                          ann_ds.h  -  description
                             -------------------
    begin                : pon kwi 14 2003
    copyright            : (C) 2003 by Bartosz Lis
    email                : bartoszl@ics.p.lodz.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __ANN_DS_H
#define __ANN_DS_H

#include <ctype.h>

#include <ann_loc.h>
#include <ann_term.h>

namespace ANN
{

  class DS;      /* data source */
  class DS_des;  /* data source for desired output */

  //          //
 // class DS //
//          //

  class DS
  {
  private:
    DS(const DS &that);

  protected:
    DS            *des_out;

    DS() : des_out(0) {}
    DS(DS *des_out_) : des_out(des_out_) {}

    virtual bool   resize(const Loc &sizes_);
    virtual bool   resize(const Size &sizes_);
    virtual bool   resize(size_t dimension_, size_t *positions_=0);
    virtual bool   resize(size_t *positions_);
    bool           set_len(size_t len_);
    virtual bool   do_reset()=0;
    virtual void   do_next()=0;
    virtual bool   do_seek(size_t term_pos_);
    virtual bool   do_set_len(size_t len_);

    bool           des_resize(const Loc &sizes_)
    { return des_out->resize(sizes_); }
    bool           des_resize(const Size &sizes_)
    { return des_out->resize(sizes_); }
    bool           des_resize(size_t dimension_, size_t *positions_=0)
    { return des_out->resize(dimension_,positions_); }
    bool           des_resize(size_t *positions_)
    { return des_out->resize(positions_); }

  public:
    virtual ~DS();

    virtual size_t      get_len() const=0; /* length of the test/training
                                            * sequence, or zero if unfinited
                                            * data stream */
    virtual const Size &get_sizes() const; /* term sizes */
    virtual size_t      get_size() const;  /* returns overall term size */
    virtual Term       &term()=0;
    virtual const Term &term() const=0;
    bool                reset();
    void                next();
    bool                seek(size_t term_pos_);
    /* the above functions call do_xxx() of objects *input() and *des_out
     * forcing both the objects to be synchronized */
    /* the real job of shifting terms is done by functions do_xxx(), that is:
     * reset()/do_reset() -- if possible shifts data source back to the first
     *                       term and returns true otherwise return false
     * next()/do_next() -- shifts data source to the next term, if the last term
     *                     was presented calling these functions should be
     *                     equivalent to calling reset()/do_reset()
     * seek()/do_seek() -- shifts data source to the specified term, never calls
     *                     reset()/do_reset(), returns true if shifting was
     *                     possible otherwise false */
    virtual size_t      where() const=0;
    /* position of the current term in the sequence, if data source randomizes
     * order of ites terms, pos returns original ordinal number while seek()
     * and do_seek() take ordinal number after terms were randomized */
    virtual Term       *all_terms();
    virtual const Term *all_terms() const;
    /* the above functions return pointer to an array of all terms making up the
     * current training sequence, if such an array can not be provided by the
     * data source, zero is returned (default behavior), in this case next()
     * reset() and seek() functions should be used */
    virtual DS         *input();
    virtual const DS   *input() const;
    /* returns data source containing inputs which is synchronized to this data
     * source (except class DS_des and its descendants, pointer this is
     * returned) */
    DS                 *des_output() { return des_out; }
    const DS           *des_output() const { return des_out; }
    /* returns data source containing desired outputs which is synchronized to
     * this data source or 0 if nothing available */
    /* the value returned by the four above functinos should point to the same
     * object if we call input source object's input() or desired output
     * object's des_output(), otherwise pointer to the other synchronised
     * obcject is returned */

    double &operator [] (size_t idx) { return term()[idx]; }
    double  operator [] (size_t idx) const { return term()[idx]; }
    double &operator [] (const Loc &loc) { return term()[loc]; }
    double  operator [] (const Loc &loc) const { return term()[loc]; }
    DS     &operator ++ () { next(); return *this; }
  };

  //              //
 // class DS_des //
//              //

  class DS_des : public virtual DS
  {
  private:
    DS_des();

  protected:
    DS   *in;

    DS_des(DS &in_) : DS(this), in(&in_) {}

  public:
    virtual ~DS_des();

    virtual size_t    get_len() const;
    virtual size_t    where() const;
    virtual DS       *input();
    virtual const DS *input() const;
  };

} // namespace ANN

#endif /* __ANN_DS_H */
