#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/uio.h>
#include <sys/queue.h>
#include <confuse.h>
#include "config.h"

#define SO_EE_ORIGIN_NONE    0
#define SO_EE_ORIGIN_LOCAL   1
#define SO_EE_ORIGIN_ICMP    2
#define SO_EE_ORIGIN_ICMP6   3

#define LISTEN_ADDR INADDR_ANY
#define LISTEN_PORT 9
#define LISTEN_BUFF_SIZE 65536
#define CONTROL_BUF_SIZE 65536

#define CFG_FILE_PATH "/etc/wold/main.cfg"

typedef unsigned char mac_addr_t[6];

#define WOL_REPETITIONS 16
#define WOL_MAGIC_LEN (sizeof(mac_addr_t)*(WOL_REPETITIONS+1))

struct sock_extended_err {
    u_int32_t ee_errno;   /* numer błędu */
    u_int8_t  ee_origin;  /* źródło błędu */
    u_int8_t  ee_type;    /* typ */
    u_int8_t  ee_code;    /* kod */
    u_int8_t  ee_pad;
    u_int32_t ee_info;    /* informacje dodatkowe */
    u_int32_t ee_data;    /* inne dane */
    /* Dalej mogą wystąpić dodatkowe informacje */
};

struct packet_info {
    struct in_addr src_addr;
    in_port_t      src_port;
    struct in_addr dst_addr;
    struct in_addr if_addr;
    unsigned int   if_index;
    //unsigned char  tgt_mac[6];
};

struct transmiter_rec {
    char                        *name;
    struct in_addr               addr;
    SLIST_ENTRY(transmiter_rec)  binding;
};
SLIST_HEAD(transmiters_head_t, transmiter_rec);
struct transmiters_head_t transmiters_head;

struct receiver_rec {
    char                      *name;
    struct in_addr             net_addr;
    struct in_addr             net_bcast;
    struct in_addr             net_mask;
    mac_addr_t                 mac;
    SLIST_ENTRY(receiver_rec)  binding;
};
SLIST_HEAD(receivers_head_t, receiver_rec);
struct receivers_head_t receivers_head;

in_port_t listen_port=0;

#define MAX_IN_ADDR_PREFIX (int)(sizeof(in_addr_t)*8)

int convert_netmask_to_prefix(struct in_addr mask)
{
    int prefix;
    static in_addr_t masks[MAX_IN_ADDR_PREFIX+1];
    static int initialized=0;
    if (!initialized)
    {
        for (prefix=0; prefix<=MAX_IN_ADDR_PREFIX; ++prefix)
            masks[prefix]=htonl(~(0xFFFFFFFF >> prefix));
        initialized=1;
    }
    for (prefix=0; prefix<=MAX_IN_ADDR_PREFIX; ++prefix)
        if (mask.s_addr==masks[prefix]) return prefix;
    return -1;
}

void print_err_va(const char *format, va_list args)
{
    vfprintf (stderr, format, args);
    fprintf (stderr, "\n");
}

void print_err(const char *format, ...)
{
    va_list args;
    va_start (args, format);
    print_err_va (format, args);
    va_end (args);
}

void init_lists()
{
    SLIST_INIT(&transmiters_head);
    SLIST_INIT(&receivers_head);
}

void free_lists()
{
    struct transmiter_rec *transmiter;
    struct receiver_rec   *receiver;

    while (!SLIST_EMPTY(&transmiters_head))
    {
        transmiter = SLIST_FIRST(&transmiters_head);
        SLIST_REMOVE_HEAD(&transmiters_head, binding);
        free(transmiter->name);
        free(transmiter);
    }
    SLIST_INIT(&transmiters_head);

    while (!SLIST_EMPTY(&receivers_head))
    {
        receiver = SLIST_FIRST(&receivers_head);
        SLIST_REMOVE_HEAD(&receivers_head, binding);
        free(receiver->name);
        free(receiver);
    }
    SLIST_INIT(&receivers_head);
}

int add_transmiter(const char *name, const struct in_addr *addr)
{
    struct transmiter_rec *rec;
    rec=(struct transmiter_rec *)malloc(sizeof(struct transmiter_rec));
    if (!rec)
    {
        print_err ("Cannot allocate memory");
        return -1;
    }
    rec->name=strdup(name);
    if (!rec->name)
    {
        print_err ("Cannot allocate memory");
        free(rec);
        return -1;
    }
    rec->addr.s_addr=addr->s_addr;
    SLIST_INSERT_HEAD(&transmiters_head, rec, binding);
    //printf("Added transmiter %s, IP: %s\n",rec->name,inet_ntoa(rec->addr));
    return 0;
}

struct transmiter_rec *find_transmiter_by_name(const char *name)
{
    struct transmiter_rec *item;
    item=SLIST_FIRST(&transmiters_head);
    while (item && strcmp(item->name,name)) item=SLIST_NEXT(item,binding);
    return item;
}

struct transmiter_rec *find_transmiter_by_addr(struct in_addr *addr)
{
    struct transmiter_rec *item;
    item=SLIST_FIRST(&transmiters_head);
    while (item && (item->addr.s_addr!=addr->s_addr)) item=SLIST_NEXT(item,binding);
    return item;
}

int add_receiver(const char *name, const struct in_addr *net_addr, unsigned int mask_prefix, mac_addr_t mac)
{
    struct receiver_rec *rec;
    rec=(struct receiver_rec *)malloc(sizeof(struct receiver_rec));
    if (!rec)
    {
        print_err ("Cannot allocate memory");
        return -1;
    }
    rec->name=strdup(name);
    if (!rec)
    {
        print_err ("Cannot allocate memory");
        free(rec);
        return -1;
    }
    rec->net_mask.s_addr=htonl(~(0xFFFFFFFF >> mask_prefix));
    rec->net_addr.s_addr=(net_addr->s_addr & rec->net_mask.s_addr);
    rec->net_bcast.s_addr=(rec->net_addr.s_addr | (0xffffffff & ~rec->net_mask.s_addr));
    memcpy(rec->mac,mac,sizeof(mac_addr_t));
    SLIST_INSERT_HEAD(&receivers_head, rec, binding);
    //printf("Added receiver %s, IP: %s",rec->name,inet_ntoa(rec->net_addr));
    //printf("/%s",inet_ntoa(rec->net_mask));
    //printf(" [%s], MAC: %02x:%02x:%02x:%02x:%02x:%02x\n",inet_ntoa(rec->net_bcast),rec->mac[0],rec->mac[1],rec->mac[2],rec->mac[3],rec->mac[4],rec->mac[5]);
    return 0;
}

struct receiver_rec *find_receiver_by_name(const char *name)
{
    struct receiver_rec *item;
    item=SLIST_FIRST(&receivers_head);
    while (item && strcmp(item->name,name)) item=SLIST_NEXT(item,binding);
    return item;
}

struct receiver_rec *find_receiver_by_mac(mac_addr_t mac)
{
    struct receiver_rec *item;
    item=SLIST_FIRST(&receivers_head);
    while (item && strncmp((char *)item->mac,(char *)mac,sizeof(mac_addr_t))) item=SLIST_NEXT(item,binding);
    return item;
}

void conf_print_err_va(cfg_t *cfg, const char *format, va_list args)
{
    char fmt[80];
    if (cfg && cfg->filename && cfg->line)
        snprintf(fmt, sizeof(fmt), "%s:%d: %s", cfg->filename, cfg->line, format);
    else if (cfg && cfg->filename)
        snprintf(fmt, sizeof(fmt), "%s: %s", cfg->filename, format);
    else
        snprintf(fmt, sizeof(fmt), "%s", format);
    print_err_va(fmt, args);
}

void conf_print_err(cfg_t *cfg, const char *format, ...)
{
    va_list args;
    va_start (args, format);
    conf_print_err_va (cfg, format, args);
    va_end (args);
}

int conf_load(void)
{
    cfg_opt_t opts_trans[] =
    {
        CFG_STR("address", 0, CFGF_NONE),
        CFG_END()
    };
    cfg_opt_t opts_recv[] =
    {
        CFG_STR("subnet", 0, CFGF_NONE),
        CFG_STR("mac",    0, CFGF_NONE),
        CFG_END()
    };
    cfg_opt_t opts[] =
    {
        CFG_INT("listen-port", LISTEN_PORT, CFGF_NONE),
        CFG_SEC("transmiter",  opts_trans,  CFGF_TITLE | CFGF_MULTI),
        CFG_SEC("receiver",    opts_recv,   CFGF_TITLE | CFGF_MULTI),
        CFG_END()
    };
    cfg_t *cfg, *cfg_trans, *cfg_recv;
    const char *trans_name, *recv_name;
    char *trans_addr_str, *recv_subnet_str, *recv_net_mask_str, *recv_mac_str, *recv_mac_tail;
    struct in_addr trans_addr, recv_net_addr, recv_net_mask;
    mac_addr_t recv_mac;
    long recv_mask_prefix;

    unsigned int i;
    cfg = cfg_init(opts, CFGF_NONE);
    if (!cfg) {
        print_err ("Failed initializing configuration file parser: %s", strerror(errno));
        return -1;
    }
    cfg_set_error_function(cfg, conf_print_err_va);
    if (cfg_parse(cfg, CFG_FILE_PATH) == CFG_PARSE_ERROR)
    {
        cfg_free(cfg);
        return -1;
    }
    listen_port=htons(cfg_getint(cfg, "listen-port"));
    for(i = 0; i < cfg_size(cfg, "transmiter"); i++)
    {
        cfg_trans = cfg_getnsec(cfg, "transmiter", i);
        trans_name = cfg_title(cfg_trans);
        if (!trans_name || !*trans_name)
        {
            conf_print_err(cfg, "Transmiter name not defined");
            cfg_free(cfg);
            return -1;
        }
        trans_addr_str = cfg_getstr(cfg_trans, "address");
        if (!trans_addr_str || !*trans_addr_str)
        {
            conf_print_err(cfg, "Transmiter %s IP address not defined", trans_name);
            cfg_free(cfg);
            return -1;
        }
        if (!inet_aton(trans_addr_str, &trans_addr))
        {
            conf_print_err(cfg, "Wrong IP address specification %s for transmiter %s", trans_addr_str, trans_name);
            cfg_free(cfg);
            return -1;
        }
        if (find_transmiter_by_name(trans_name))
        {
            conf_print_err(cfg, "Duplicate transmiter %s definition", trans_name);
            cfg_free(cfg);
            return -1;
        }
        if (find_transmiter_by_addr(&trans_addr))
        {
            conf_print_err(cfg, "Duplicate IP address %s in transmiter %s definition", trans_addr_str, trans_name);
            cfg_free(cfg);
            return -1;
        }
        if (add_transmiter(trans_name,&trans_addr)==-1)
        {
            cfg_free(cfg);
            return -1;
        }
    }
    for(i = 0; i < cfg_size(cfg, "receiver"); i++)
    {
        cfg_recv = cfg_getnsec(cfg, "receiver", i);
        recv_name = cfg_title(cfg_recv);
        if (!recv_name || !*recv_name)
        {
            conf_print_err(cfg, "Receiver name not defined");
            cfg_free(cfg);
            return -1;
        }
        recv_subnet_str = cfg_getstr(cfg_recv, "subnet");
        if (!recv_subnet_str || !*recv_subnet_str)
        {
            conf_print_err(cfg, "Receiver %s IP subnet not defined", recv_name);
            cfg_free(cfg);
            return -1;
        }
        recv_mac_str = cfg_getstr(cfg_recv, "mac");
        if (!recv_mac_str || !*recv_mac_str)
        {
            conf_print_err(cfg, "Receiver %s MAC address not defined", recv_name);
            cfg_free(cfg);
            return -1;
        }
        recv_net_mask_str=strstr(recv_subnet_str,"/");
        if(!recv_net_mask_str)
        {
            conf_print_err(cfg, "Cannot parse receiver %s subnet specification %s", recv_name, recv_subnet_str);
            cfg_free(cfg);
            return -1;
        }
        *(recv_net_mask_str++)=0;
        if (!inet_aton(recv_subnet_str, &recv_net_addr))
        {
            conf_print_err(cfg, "Wrong subnet IP address specification %s for receiver %s", recv_subnet_str, recv_name);
            cfg_free(cfg);
            return -1;
        }
        recv_mask_prefix=strtol(recv_net_mask_str,&recv_mac_tail,10);
        if (*recv_mac_tail)
        {
            if (!inet_aton(recv_net_mask_str, &recv_net_mask))
            {
                conf_print_err(cfg, "Wrong subnet mask specification %s for receiver %s", recv_net_mask_str, recv_name);
                cfg_free(cfg);
                return -1;
            }
            recv_mask_prefix=convert_netmask_to_prefix(recv_net_mask);
            if (recv_mask_prefix<0)
            {
                conf_print_err(cfg, "Wrong subnet mask specification %s for receiver %s", recv_net_mask_str, recv_name);
                cfg_free(cfg);
                return -1;
            }
        }
        else if (recv_mask_prefix<0)
        {
            conf_print_err(cfg, "Subnet prefix %s for receiver %s should not be negative", recv_net_mask_str, recv_name);
            cfg_free(cfg);
            return -1;
        }
        else if (recv_mask_prefix>MAX_IN_ADDR_PREFIX)
        {
            conf_print_err(cfg, "Subnet prefix %s for receiver %s should not be greater than %d", recv_net_mask_str, recv_name, MAX_IN_ADDR_PREFIX);
            cfg_free(cfg);
            return -1;
        }
        if ( 6 != sscanf(recv_mac_str, "%2hhx:%2hhx:%2hhx:%2hhx:%2hhx:%2hhx%*c", &recv_mac[0], &recv_mac[1], &recv_mac[2], &recv_mac[3], &recv_mac[4], &recv_mac[5] ))
        {
            conf_print_err(cfg, "Cannot parse mac address %s for receiver %s", recv_mac_str, recv_name);
            cfg_free(cfg);
            return -1;
        }
        if (find_receiver_by_name(recv_name))
        {
            conf_print_err(cfg, "Duplicate receiver %s definition", recv_name);
            cfg_free(cfg);
            return -1;
        }
        if (find_receiver_by_mac(recv_mac))
        {
            conf_print_err(cfg, "Duplicate MAC address %s in receiver %s definition", recv_mac_str, recv_name);
            cfg_free(cfg);
            return -1;
        }
        if (add_receiver(recv_name,&recv_net_addr,recv_mask_prefix,recv_mac)==-1)
        {
            cfg_free(cfg);
            return -1;
        }
    }
    cfg_free(cfg);
    return 0;
}

void conf_dump(FILE *file)
{
    struct transmiter_rec *transmiter;
    struct receiver_rec *receiver;
    fprintf(file,"listen-port = %d\n",ntohs(listen_port));
    for (transmiter=SLIST_FIRST(&transmiters_head); transmiter; transmiter=SLIST_NEXT(transmiter,binding))
    {
        fprintf(file,"transmiter %s {\n",transmiter->name);
        fprintf(file,"\taddress = %s\n",inet_ntoa(transmiter->addr));
        fprintf(file,"}\n");
    }
    for (receiver=SLIST_FIRST(&receivers_head); receiver; receiver=SLIST_NEXT(receiver,binding))
    {
        fprintf(file,"receiver %s {\n",receiver->name);
        fprintf(file,"\tsubnet = %s/%d\n",inet_ntoa(receiver->net_addr),convert_netmask_to_prefix(receiver->net_mask));
        fprintf(file,"\tmac = %02x:%02x:%02x:%02x:%02x:%02x\n",receiver->mac[0],receiver->mac[1],receiver->mac[2],receiver->mac[3],receiver->mac[4],receiver->mac[5]);
        fprintf(file,"}\n");
    }
}

int listen_udp_port()
{
    struct sockaddr_in saddr;
    int opt, ret;
    memset(&saddr, 0, sizeof(struct sockaddr_in));
    saddr.sin_family = AF_INET;
    saddr.sin_addr.s_addr = htonl(LISTEN_ADDR);
    saddr.sin_port = listen_port;
    int fd=socket(AF_INET,SOCK_DGRAM,0);
    if (fd==-1) {
        print_err ("failed to open socket (#%d): %s", errno, strerror(errno));
        return -1;
    }
    opt=1;
    ret=setsockopt(fd, IPPROTO_IP, IP_PKTINFO, &opt, sizeof(opt));
    if (ret==-1) {
        close(fd);
        print_err ("failed to set socket option (#%d): %s", errno, strerror(errno));
        return -1;
    }
    ret=bind(fd,(struct sockaddr *)&saddr,sizeof(struct sockaddr_in));
    if (ret==-1) {
        close(fd);
        print_err ("failed to bind socket (#%d): %s", errno, strerror(errno));
        return -1;
    }
    return fd;
}

ssize_t receive_packet(int in_sock, char *buff, ssize_t buff_len, struct packet_info *info)
{
    struct msghdr             recv_msg;
    struct iovec              iov[1];
    struct sockaddr_in        src;
    char                      control[CONTROL_BUF_SIZE];
    ssize_t                   packet_size;
    struct cmsghdr           *cmsg;
    struct in_pktinfo        *pi;
    struct sock_extended_err *ee;

    memset(info,0,sizeof(struct packet_info));
    memset(control,0,CONTROL_BUF_SIZE);
    iov[0].iov_base = buff;
    iov[0].iov_len = buff_len;
    recv_msg.msg_name=&src;
    recv_msg.msg_namelen=sizeof(struct sockaddr_in);
    recv_msg.msg_iov=iov;
    recv_msg.msg_iovlen=1;
    recv_msg.msg_control=&control;
    recv_msg.msg_controllen=CONTROL_BUF_SIZE;
    recv_msg.msg_flags=0;
//    packet_size = recvmsg(in_sock, &recv_msg, MSG_ERRQUEUE);
    packet_size = recvmsg(in_sock, &recv_msg, 0);
    if (packet_size==-1) {
        switch (errno)
        {
        case EINTR:
        case EAGAIN:
            packet_size=0;
            break;
        default:
            print_err ("failed to receive packet (#%d): %s", errno, strerror(errno));
            return -1;
        }
    }
    if (recv_msg.msg_controllen>0) for (cmsg = CMSG_FIRSTHDR(&recv_msg); cmsg != NULL; cmsg = CMSG_NXTHDR(&recv_msg, cmsg)) {
        printf("cmsg_len=%zu, cmsg_level=%u, cmsg_type=%u\n", cmsg->cmsg_len, cmsg->cmsg_level, cmsg->cmsg_type);
        switch (cmsg->cmsg_level)
        {
        case IPPROTO_IP:
            switch (cmsg->cmsg_type)
            {
            case IP_PKTINFO:
                pi = (struct in_pktinfo *)CMSG_DATA(cmsg);
                info->dst_addr.s_addr=pi->ipi_addr.s_addr;
                info->if_addr.s_addr=pi->ipi_spec_dst.s_addr;
                info->if_index=pi->ipi_ifindex;
                break;
            case IP_RECVERR:
                ee=(struct sock_extended_err *)CMSG_DATA(cmsg);
                print_err ("failed to receive packet (#%d): %s", ee->ee_errno, strerror(ee->ee_errno));
                break;
            }
            break;
        }
    }
    info->src_addr.s_addr=src.sin_addr.s_addr;
    info->src_port=src.sin_port;
    return packet_size;
}

int wol_send_raw(struct receiver_rec *receiver, unsigned char *packet, ssize_t packet_len)
{
    struct sockaddr_in saddr;
    int out_sock;
    ssize_t ret;
    int broadcastEnable=1;
    memset(&saddr, 0, sizeof(struct sockaddr_in));
    saddr.sin_family = AF_INET;
    saddr.sin_addr.s_addr = receiver->net_bcast.s_addr;
    saddr.sin_port = listen_port;

    out_sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (out_sock == -1)
    {
        print_err ("failed to open outgoing socket (#%d): %s", errno, strerror(errno));
        return -1;
    }
    ret=setsockopt(out_sock, SOL_SOCKET, SO_BROADCAST, &broadcastEnable, sizeof(broadcastEnable));
    if (ret == -1)
    {
        close(out_sock);
        print_err ("failed to set socket option (#%d): %s", errno, strerror(errno));
        return -1;
    }
    ret=sendto(out_sock,packet,packet_len,0,(struct sockaddr *) &saddr, sizeof(struct sockaddr_in));
    if (ret == -1)
    {
        close(out_sock);
        print_err ("cannot send packet (#%d): %s", errno, strerror(errno));
        return -1;
    }
    close(out_sock);
    return 0;
}

void process_packet(unsigned char *packet, ssize_t packet_len, struct packet_info *info)
{
    mac_addr_t mac;
    static mac_addr_t magic_prefix = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
    struct transmiter_rec *transmiter;
    struct receiver_rec *receiver;
    unsigned char *base;
    int i;

    printf("Got packet size: %zd\n",packet_len);
    printf("\tsource:      %s:%u\n",inet_ntoa(info->src_addr),info->src_port);
    printf("\tdestination: %s\n",inet_ntoa(info->dst_addr));
    printf("\treceived on: %s at interface #%u\n",inet_ntoa(info->if_addr),info->if_index);

    transmiter=find_transmiter_by_addr(&info->src_addr);
    if (!transmiter)
    {
        printf("\tundeclared transmiter, packet dropped.\n");
        return;
    }

    base=packet;
    while (packet_len>=(ssize_t)WOL_MAGIC_LEN)
    {
        if ((*base == 0xff) && !strncmp((const char *)base,(const char *)magic_prefix,sizeof(mac_addr_t)))
        {
            memcpy(mac,base+sizeof(mac_addr_t),sizeof(mac_addr_t));
            for (i=2; (i<=WOL_REPETITIONS) && !strncmp((const char *)(base+i*sizeof(mac_addr_t)),(const char *)mac,sizeof(mac_addr_t)) ; ++i);
            if (i>WOL_REPETITIONS)
            {
                packet=base;
                packet_len-=WOL_MAGIC_LEN;
                base+=WOL_MAGIC_LEN;
                printf("\tmac:         %02x:%02x:%02x:%02x:%02x:%02x\n",mac[0],mac[1],mac[2],mac[3],mac[4],mac[5]);
                receiver=find_receiver_by_mac(mac);
                if (!receiver)
                {
                    printf("\tundeclared receiver, skipped.\n");
                    continue;
                }
                if (~info->dst_addr.s_addr!=0)
                {
                	if (receiver->net_addr.s_addr != (info->dst_addr.s_addr & receiver->net_mask.s_addr))
                	{
                		printf("\treceiver network address mismatch: %s",inet_ntoa(receiver->net_addr));
                		printf(" <-> %s, skipped.\n",inet_ntoa(info->dst_addr));
                		continue;
                	}
                	if (receiver->net_bcast.s_addr != info->dst_addr.s_addr)
                	{
                		printf("\treceiver network broadcast address mismatch, skipped.\n");
                		continue;
                	}
                }
                wol_send_raw(receiver, packet, WOL_MAGIC_LEN);
                continue;
            }
        }
        ++base;
        --packet_len;
    }
}

int main()
{
    int in_sock;
    unsigned char *in_buff;
    ssize_t in_packet_size;
    struct packet_info info;
    in_buff=(unsigned char *)malloc(LISTEN_BUFF_SIZE);
    if (!in_buff)
    {
        print_err ("cannot allocate memory");
        exit(13);
    }
    if (conf_load() == -1) return 0;
    conf_dump(stdout);
    if ((in_sock = listen_udp_port())==-1) exit(2);
    while ((in_packet_size=receive_packet(in_sock,in_buff,LISTEN_BUFF_SIZE,&info))!=-1)
        if (in_packet_size>0) process_packet(in_buff,in_packet_size,&info);
    return 0;
}
